document.addEventListener('DOMContentLoaded', function () {
  const subtotalEl = document.getElementById('subtotal');
  const totalEl = document.getElementById('total');
  const discountEl = document.getElementById('discount');
  const summaryList = document.getElementById('summaryList');
  const resetBtn = document.getElementById('resetBtn');

  function money(v) {
    return '$' + v.toFixed(2);
  }

  function calculate() {
    let subtotal = 0;
    summaryList.innerHTML = '';
    let freqDiscount = 0;

    // --- SELECTS (bedrooms, bathrooms, sqft, zipcode) ---
    document.querySelectorAll('select.calc-field').forEach(sel => {
      const selected = sel.options[sel.selectedIndex];
      if (!selected || !sel.value) return;

      const price = parseFloat(selected.dataset.price) || 0;
      subtotal += price;

      const categoryName =
        sel.dataset.category.charAt(0).toUpperCase() +
        sel.dataset.category.slice(1);
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${categoryName}: ${selected.text}</div><div>${money(price)}</div>`;
      summaryList.appendChild(li);
    });

    // --- FREQUENCY RADIOS ---
    const freq = document.querySelector('input.calc-field[name="frequency"]:checked');
    if (freq) {
      freqDiscount = parseFloat(freq.dataset.price) || 0;
      const label = document.querySelector(`label[for="${freq.id}"]`)?.innerText || 'Frequency';
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${label}</div><div><span class="badge-freq">-${freqDiscount}%</span></div>`;
      summaryList.appendChild(li);
    }

    // --- EXTRAS CHECKBOXES WITH QUANTITY ---
    let extrasSum = 0;
    document.querySelectorAll('.extras-box-input:checked').forEach(chk => {
      const price = parseFloat(chk.dataset.price) || 0;
      const box = chk.closest('.extras-box');
      const qty = parseInt(box.querySelector('.qty-input')?.value) || 1;
      const totalPrice = price * qty;
      extrasSum += totalPrice;

      const label = document.querySelector(`label[for="${chk.id}"]`)?.innerText || 'Extra';
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${label}</div><div><span class="badge-extra">${money(totalPrice)}</span> x${qty}</div>`;
      summaryList.appendChild(li);
    });

    subtotal += extrasSum;
    subtotalEl.innerText = money(subtotal);

    const discountAmount = subtotal * (freqDiscount / 100);
    discountEl.innerText = '-' + money(discountAmount);
    totalEl.innerText = money(subtotal - discountAmount);
  }

  // --- EVENT LISTENERS ---
  document.querySelectorAll('select.calc-field').forEach(sel => sel.addEventListener('change', calculate));
  document.querySelectorAll('input.calc-field[name="frequency"]').forEach(radio => radio.addEventListener('change', calculate));

  // Extras checkboxes with quantity handling
  document.querySelectorAll('.extras-box-input').forEach(chk => {
    const box = chk.closest('.extras-box');
    const qtySelector = box.querySelector('.quantity-selector');
    const qtyInput = box.querySelector('.qty-input');
    const minusBtn = box.querySelector('.qty-minus');
    const plusBtn = box.querySelector('.qty-plus');
    const maxQty = parseInt(chk.dataset.max) || 10;

    chk.addEventListener('change', () => {
      if (chk.checked) {
        box.classList.add('active');
        qtySelector.style.display = 'flex';
        qtySelector.style.opacity = '1';
        if (parseInt(qtyInput.value) < 1) qtyInput.value = 1;
      } else {
        box.classList.remove('active');
        qtySelector.style.opacity = '0';
        setTimeout(() => { qtySelector.style.display = 'none'; }, 200);
      }
      calculate();
    });

    plusBtn.addEventListener('click', () => {
      let val = parseInt(qtyInput.value);
      if (val < maxQty) {
        qtyInput.value = val + 1;
        calculate();
      }
    });

    minusBtn.addEventListener('click', () => {
      qtyInput.value = Math.max(1, parseInt(qtyInput.value) - 1);
      calculate();
    });

    qtyInput.addEventListener('input', () => {
      if (qtyInput.value < 1) qtyInput.value = 1;
      if (qtyInput.value > maxQty) qtyInput.value = maxQty;
      calculate();
    });
  });

  // Reset button
  resetBtn.addEventListener('click', function () {
    document.querySelectorAll('select.calc-field').forEach(sel => sel.selectedIndex = 0);
    document.querySelectorAll('input.calc-field[name="frequency"]').forEach(radio => radio.checked = false);
    document.querySelectorAll('.extras-box-input').forEach(chk => {
      chk.checked = false;
      const box = chk.closest('.extras-box');
      const qtySelector = box.querySelector('.quantity-selector');
      box.classList.remove('active');
      qtySelector.style.display = 'none';
      qtySelector.style.opacity = '0';
      box.querySelector('.qty-input').value = 1;
    });
    calculate();
  });

  calculate();
});
