document.addEventListener('DOMContentLoaded', function () {
  const subtotalEl = document.getElementById('subtotal');
  const totalEl = document.getElementById('total');
  const discountEl = document.getElementById('discount');
  const summaryList = document.getElementById('summaryList');
  const resetBtn = document.getElementById('resetBtn');
  const partialToggle = document.getElementById('partialToggle');
  const partialContainer = document.querySelector('.partial-items-container');

  function money(v) { return '$' + v.toFixed(2); }

  // Show/hide partial cleaning
  if (partialToggle && partialContainer) {
    partialToggle.addEventListener('change', () => {
      partialContainer.style.display = partialToggle.checked ? 'flex' : 'none';
      equalizeHeights('.partial-items-container .extras-box');
      calculate();
    });
  }

  function calculate() {
    let subtotal = 0;
    summaryList.innerHTML = '';
    let freqDiscount = 0;
    const listedItems = new Set();

    // --- SELECTS ---
    document.querySelectorAll('select.calc-field').forEach(sel => {
      const selected = sel.options[sel.selectedIndex];
      if (!selected || !sel.value) return;
      const price = parseFloat(selected.dataset.price) || 0;
      subtotal += price;

      const categoryName = sel.dataset.category.charAt(0).toUpperCase() + sel.dataset.category.slice(1);
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${categoryName}: ${selected.text}</div><div>${money(price)}</div>`;
      summaryList.appendChild(li);

      listedItems.add('select-' + sel.value);
    });

    // --- FREQUENCY ---
    const freq = document.querySelector('input.calc-field[name="frequency"]:checked');
    if (freq) {
      freqDiscount = parseFloat(freq.dataset.price) || 0;
      const label = document.querySelector(`label[for="${freq.id}"]`)?.innerText || 'Frequency';
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${label}</div><div><span class="badge-freq">-${freqDiscount}%</span></div>`;
      summaryList.appendChild(li);
    }

    // --- EXTRAS ---
    subtotal += processBoxes('.extras-container', 'Extras', listedItems, false);

    // --- PARTIAL CLEANING ---
    if (partialToggle?.checked) {
      subtotal -= processBoxes('.partial-items-container', 'Partial Cleaning', listedItems, true);
    }

    subtotalEl.innerText = money(subtotal);
    const discountAmount = subtotal * (freqDiscount / 100);
    discountEl.innerText = '-' + money(discountAmount);
    totalEl.innerText = money(subtotal - discountAmount);
    // --- Update floating header total ---
    if (summaryHeaderTotal) summaryHeaderTotal.textContent = totalEl.textContent;


    // --- Price range display (dynamic margin from settings, rounded to nearest $5) ---
    if (typeof PRICE_RANGE_MARGIN !== 'undefined') {
      const totalValue = subtotal - discountAmount;

      const lower = totalValue * (1 - PRICE_RANGE_MARGIN / 100);
      const upper = totalValue * (1 + PRICE_RANGE_MARGIN / 100);

      // Round both to nearest $5
      const roundToFive = (val) => Math.round(val / 5) * 5;

      const min = roundToFive(lower);
      const max = roundToFive(upper);

      const rangeLowEl = document.getElementById('rangeLow');
      const rangeHighEl = document.getElementById('rangeHigh');

      if (rangeLowEl && rangeHighEl) {
        if (totalValue <= 0) {
          rangeLowEl.innerText = '$0.00';
          rangeHighEl.innerText = '$0.00';
        } else {
          rangeLowEl.innerText = money(min);
          rangeHighEl.innerText = money(max);
        }
      }
    }

  }

  function processBoxes(containerSelector, groupName, listedItems, isPartial = false) {
    let sum = 0;
    const container = document.querySelector(containerSelector);
    if (!container) return 0;

    const checkedBoxes = container.querySelectorAll('.extras-box-input:checked');
    if (checkedBoxes.length > 0 && groupName) {
      const liGroup = document.createElement('li');
      liGroup.className = 'list-group-item list-group-item-secondary fw-bold';
      liGroup.innerText = groupName;
      summaryList.appendChild(liGroup);
    }

    checkedBoxes.forEach(chk => {
      const itemKey = (isPartial ? 'partial-' : 'extra-') + chk.dataset.itemId;
      if (listedItems.has(itemKey)) return;
      listedItems.add(itemKey);

      const price = parseFloat(chk.dataset.price) || 0;
      const box = chk.closest('.extras-box');
      const qty = parseInt(box.querySelector('.qty-input')?.value) || 1;
      const totalPrice = price * qty;
      sum += totalPrice;

      const label = document.querySelector(`label[for="${chk.id}"]`)?.innerText || 'Item';
      const li = document.createElement('li');
      li.className = 'list-group-item d-flex justify-content-between';
      li.innerHTML = `<div>${label}</div><div><span class="${isPartial ? 'badge-partial' : 'badge-extra'}">${money(totalPrice)}</span> x${qty}</div>`;
      summaryList.appendChild(li);
    });

    return sum;
  }

  // --- Inputs & quantity handling ---
  document.querySelectorAll('select.calc-field').forEach(sel => sel.addEventListener('change', calculate));
  document.querySelectorAll('input.calc-field[name="frequency"]').forEach(r => r.addEventListener('change', calculate));
  document.querySelectorAll('.extras-box-input').forEach(chk => {
    const box = chk.closest('.extras-box');
    const qtySelector = box.querySelector('.quantity-selector');
    const qtyInput = box.querySelector('.qty-input');
    const minusBtn = box.querySelector('.qty-minus');
    const plusBtn = box.querySelector('.qty-plus');
    const maxQty = parseInt(chk.dataset.max) || 10;

    chk.addEventListener('change', () => {
      box.classList.toggle('active', chk.checked);
      qtySelector.style.display = chk.checked ? 'flex' : 'none';
      calculate();
    });
    plusBtn.addEventListener('click', () => {
      if (parseInt(qtyInput.value) < maxQty) qtyInput.value = parseInt(qtyInput.value) + 1;
      calculate();
    });
    minusBtn.addEventListener('click', () => {
      qtyInput.value = Math.max(1, parseInt(qtyInput.value) - 1);
      calculate();
    });
    qtyInput.addEventListener('input', () => {
      if (qtyInput.value < 1) qtyInput.value = 1;
      if (qtyInput.value > maxQty) qtyInput.value = maxQty;
      calculate();
    });
  });

  // --- Reset ---
  resetBtn.addEventListener('click', () => {
    document.querySelectorAll('select.calc-field').forEach(sel => sel.selectedIndex = 0);
    document.querySelectorAll('input.calc-field[name="frequency"]').forEach(r => r.checked = false);
    if (partialToggle) partialToggle.checked = false;
    if (partialContainer) partialContainer.style.display = 'none';
    document.querySelectorAll('.extras-box-input').forEach(chk => {
      chk.checked = false;
      const box = chk.closest('.extras-box');
      box.classList.remove('active');
      const qtySelector = box.querySelector('.quantity-selector');
      qtySelector.style.display = 'none';
      box.querySelector('.qty-input').value = 1;
    });
    calculate();
  });

  // --- Equal heights for partial boxes ---
  function equalizeHeights(selector) {
    const boxes = document.querySelectorAll(selector);
    if (!boxes.length) return;
    let maxHeight = 0;
    boxes.forEach(box => box.style.height = 'auto');
    boxes.forEach(box => { if (box.offsetHeight > maxHeight) maxHeight = box.offsetHeight; });
    boxes.forEach(box => box.style.height = maxHeight + 'px');
  }

  equalizeHeights('.partial-items-container .extras-box');
  window.addEventListener('resize', () => equalizeHeights('.partial-items-container .extras-box'));


  // --- Floating Summary Toggle (Mobile) ---
  const summaryCard = document.getElementById('summaryCard');
  const summaryHeader = summaryCard?.querySelector('.summary-header');
  const summaryHeaderText = document.getElementById('summaryHeaderText');
  const summaryHeaderTotal = document.getElementById('summaryHeaderTotal');

  if (summaryCard) {
    summaryCard.addEventListener('click', () => {
      summaryCard.classList.toggle('expanded');
      summaryHeaderText.innerText = summaryCard.classList.contains('expanded')
        ? 'Hide Summary'
        : 'View Summary';
    });
  }


  
  // --- Set default selections ---
  /*
  const defaultZip = document.querySelector('select.calc-field[data-category="zipcodes"] option[value="77"]'); // 33496
  if (defaultZip) defaultZip.selected = true;

  const firstBedrooms = document.querySelector('select.calc-field[data-category="bedrooms"] option:nth-child(2)'); // first real option
  if (firstBedrooms) firstBedrooms.selected = true;

  const firstBathrooms = document.querySelector('select.calc-field[data-category="bathrooms"] option:nth-child(2)');
  if (firstBathrooms) firstBathrooms.selected = true;

  const firstSqft = document.querySelector('select.calc-field[data-category="square-foot-range"] option:nth-child(2)');
  if (firstSqft) firstSqft.selected = true;

  const oneTimeFreq = document.querySelector('input.calc-field[name="frequency"][id="freq-87"]'); // One-Time
  if (oneTimeFreq) oneTimeFreq.checked = true;
  */

  // --- Initial calculation ---
  calculate();
});
